﻿//|---------------------------------------------------------------|
//|                         SORT LIBRARY                          |
//|                    Developed by Wonde Tadesse                 |  
//|                             Copyright ©  2011                 |
//|---------------------------------------------------------------|
//|                         SORT LIBRARY                          |
//|---------------------------------------------------------------|
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;

namespace SortLibray
{
    /// <summary>
    /// Character Comparer class
    /// </summary>
    /// <typeparam name="T">A Type T which it's property is being compared</typeparam>
    public class CharacterComparer<T> : IComparer<T>
    {
        #region Public Properties

        /// <summary>
        /// The PropertyName used for comparing the type. Must be contained in the type T
        /// </summary>
        private string PropertyName { get; set; }

        /// <summary>
        /// Get or set an indicator used to Compare values considering as case sensitive. True means consider as case sensitive
        /// </summary>
        public bool CaseSensitive { get; set; }

        #endregion

        #region Constructors

        /// <summary>
        /// Character Comparer class
        /// </summary>
        /// <param name="propertyName">PropertyName used for comparing the type. Must be contained in the type T</param>
        /// <remarks></remarks>
        public CharacterComparer(string propertyName)
        {
            this.PropertyName = propertyName;
            this.CaseSensitive = false;
        }

        /// <summary>
        /// Character Comparer class
        /// </summary>
        /// <param name="propertyName">PropertyName used for comparing the type. Must be contained in the Type T</param>
        /// <param name="caseSensitive">An Indicator used to Compare values considering as case sensitive.True means consider as case-sensitive</param>
        public CharacterComparer(string propertyName, bool caseSensitive)
        {
            this.PropertyName = propertyName;
            this.CaseSensitive = caseSensitive;
        }

        #endregion

        #region IComparer<T> Members

        /// <summary>
        /// Compares the left and right property values of a Type T
        /// </summary>
        /// <param name="left">Left value of Type T</param>
        /// <param name="right">Right value of Type T</param>
        /// <returns>An indicator whether the left or right is greater or not</returns>
        /// <remarks></remarks>
        public int Compare(T left, T right)
        {
            PropertyInfo leftProperty = null, rightProperty = null;

            try
            {
                // Check a PropertyName is exists.
                if (((PropertyName != null) || !string.IsNullOrEmpty(PropertyName)) & (leftProperty == null))
                {
                    leftProperty = left.GetType().GetProperty(PropertyName);
                    if (leftProperty == null)
                        throw new Exception(" Type '" + left.ToString() + "' does not contain '" + PropertyName + "' property name !");
                    rightProperty = right.GetType().GetProperty(PropertyName);
                }

                string leftValue = string.Empty, rightValue = string.Empty;

                if (leftProperty.GetValue(left, null) != null) // If a value of the property is null, set empty string
                    leftValue = Convert.ToString(leftProperty.GetValue(left, null));

                if (rightProperty.GetValue(right, null) != null)
                    rightValue = Convert.ToString(rightProperty.GetValue(right, null));

                if (string.IsNullOrEmpty(leftValue) || string.IsNullOrEmpty(rightValue)) 
                    return -1;

                if (!CaseSensitive) // Make values case-insensitive
                {
                    leftValue = leftValue.ToLowerInvariant();
                    rightValue = rightValue.ToLowerInvariant();
                }

                int lengthLeft = leftValue.Length, lengthRight = rightValue.Length, indicatorLeft = 0, indicatorRight = 0;

                while (indicatorLeft < lengthLeft && indicatorRight < lengthRight)
                {
                    char charLeft = leftValue[indicatorLeft]; // Traverse each character of the left and right values of Type T
                    char charRight = rightValue[indicatorRight];

                    char[] spaceLeft = new char[lengthLeft]; // Allocate a char array, based on the left and right values length of Type T
                    char[] spaceRight = new char[lengthRight];
                    int lockerLeft = 0,lockerRight = 0;

                    do // Iterate each characters of the left value of the Type T , until you get a Digit 
                    {
                        spaceLeft[lockerLeft] = charLeft;
                        lockerLeft = lockerLeft + 1;
                        indicatorLeft = indicatorLeft + 1;

                        if (indicatorLeft < lengthLeft)
                            charLeft = leftValue[indicatorLeft];
                        else
                            break;

                    } while (char.IsDigit(charLeft) == char.IsDigit(spaceLeft[0]));

                    do // Iterate each characters of the right value of the Type T , until you get a Digit 
                    {
                        spaceRight[lockerRight] = charRight;
                        lockerRight = lockerRight + 1;
                        indicatorRight = indicatorRight + 1;

                        if (indicatorRight < lengthRight)
                            charRight = rightValue[indicatorRight];
                        else
                            break;

                    } while (char.IsDigit(charRight) == char.IsDigit(spaceRight[0]));

                    string resultLeft = new string(spaceLeft); // Get the result from the loops
                    string resultRight = new string(spaceRight);

                    int result = 0;

                    if (char.IsDigit(spaceLeft[0]) && char.IsDigit(spaceRight[0])) // If the left and the right characters are digits then compare them as a number
                    {
                        int numberLeft = int.Parse(resultLeft);
                        int numberRight = int.Parse(resultRight);
                        result = numberLeft.CompareTo(numberRight);
                    }
                    else // Compare them as a string
                        result = resultLeft.CompareTo(resultRight);

                    if (result != 0)
                        return result;

                }
                return lengthLeft - lengthRight;
            }
            catch (Exception exception)
            {
                throw exception;
            }
        }

        #endregion

    }
}
